/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.crypto;

import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Comparator;
import javax.annotation.Nullable;
import org.bitcoinj.core.Address;
import org.bitcoinj.core.Base58;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.Utils;
import org.bitcoinj.crypto.ChildNumber;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.HDKeyDerivation;
import org.bitcoinj.crypto.HDUtils;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.script.Script;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.math.ec.ECPoint;

public class DeterministicKey
extends ECKey {
    public static final Comparator<ECKey> CHILDNUM_ORDER = new Comparator<ECKey>(){

        @Override
        public int compare(ECKey k1, ECKey k2) {
            ChildNumber cn1 = ((DeterministicKey)k1).getChildNumber();
            ChildNumber cn2 = ((DeterministicKey)k2).getChildNumber();
            return cn1.compareTo(cn2);
        }
    };
    private final DeterministicKey parent;
    private final ImmutableList<ChildNumber> childNumberPath;
    private final int depth;
    private int parentFingerprint;
    private final byte[] chainCode;

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, LazyECPoint publicAsPoint, @Nullable BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, DeterministicKey.compressPoint((LazyECPoint)Preconditions.checkNotNull((Object)publicAsPoint)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = parent == null ? 0 : parent.depth + 1;
        this.parentFingerprint = parent != null ? parent.getFingerprint() : 0;
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, ECPoint publicAsPoint, @Nullable BigInteger priv, @Nullable DeterministicKey parent) {
        this(childNumberPath, chainCode, new LazyECPoint(publicAsPoint), priv, parent);
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, BigInteger priv, @Nullable DeterministicKey parent) {
        super(priv, DeterministicKey.compressPoint(ECKey.publicPointFromPrivate(priv)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = parent == null ? 0 : parent.depth + 1;
        this.parentFingerprint = parent != null ? parent.getFingerprint() : 0;
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, KeyCrypter crypter, LazyECPoint pub, EncryptedData priv, @Nullable DeterministicKey parent) {
        this(childNumberPath, chainCode, pub, null, parent);
        this.encryptedPrivateKey = (EncryptedData)Preconditions.checkNotNull((Object)priv);
        this.keyCrypter = (KeyCrypter)Preconditions.checkNotNull((Object)crypter);
    }

    private int ascertainParentFingerprint(DeterministicKey parentKey, int parentFingerprint) throws IllegalArgumentException {
        if (parentFingerprint != 0) {
            if (this.parent != null) {
                Preconditions.checkArgument((this.parent.getFingerprint() == parentFingerprint ? 1 : 0) != 0, (String)"parent fingerprint mismatch", (Object)Integer.toHexString(this.parent.getFingerprint()), (Object)Integer.toHexString(parentFingerprint));
            }
            return parentFingerprint;
        }
        return 0;
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, LazyECPoint publicAsPoint, @Nullable DeterministicKey parent, int depth, int parentFingerprint) {
        super(null, DeterministicKey.compressPoint((LazyECPoint)Preconditions.checkNotNull((Object)publicAsPoint)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = depth;
        this.parentFingerprint = this.ascertainParentFingerprint(parent, parentFingerprint);
    }

    public DeterministicKey(ImmutableList<ChildNumber> childNumberPath, byte[] chainCode, BigInteger priv, @Nullable DeterministicKey parent, int depth, int parentFingerprint) {
        super(priv, DeterministicKey.compressPoint(ECKey.publicPointFromPrivate(priv)));
        Preconditions.checkArgument((chainCode.length == 32 ? 1 : 0) != 0);
        this.parent = parent;
        this.childNumberPath = (ImmutableList)Preconditions.checkNotNull(childNumberPath);
        this.chainCode = Arrays.copyOf(chainCode, chainCode.length);
        this.depth = depth;
        this.parentFingerprint = this.ascertainParentFingerprint(parent, parentFingerprint);
    }

    public DeterministicKey(DeterministicKey keyToClone, DeterministicKey newParent) {
        super(keyToClone.priv, keyToClone.pub.get());
        this.parent = newParent;
        this.childNumberPath = keyToClone.childNumberPath;
        this.chainCode = keyToClone.chainCode;
        this.encryptedPrivateKey = keyToClone.encryptedPrivateKey;
        this.depth = this.childNumberPath.size();
        this.parentFingerprint = this.parent.getFingerprint();
    }

    public ImmutableList<ChildNumber> getPath() {
        return this.childNumberPath;
    }

    public String getPathAsString() {
        return HDUtils.formatPath(this.getPath());
    }

    public int getDepth() {
        return this.depth;
    }

    public ChildNumber getChildNumber() {
        return this.childNumberPath.size() == 0 ? ChildNumber.ZERO : (ChildNumber)this.childNumberPath.get(this.childNumberPath.size() - 1);
    }

    public byte[] getChainCode() {
        return this.chainCode;
    }

    public byte[] getIdentifier() {
        return Utils.sha256hash160(this.getPubKey());
    }

    public int getFingerprint() {
        return ByteBuffer.wrap(Arrays.copyOfRange(this.getIdentifier(), 0, 4)).getInt();
    }

    @Nullable
    public DeterministicKey getParent() {
        return this.parent;
    }

    public int getParentFingerprint() {
        return this.parentFingerprint;
    }

    public byte[] getPrivKeyBytes33() {
        byte[] bytes33 = new byte[33];
        byte[] priv = this.getPrivKeyBytes();
        System.arraycopy(priv, 0, bytes33, 33 - priv.length, priv.length);
        return bytes33;
    }

    public DeterministicKey dropPrivateBytes() {
        if (this.isPubKeyOnly()) {
            return this;
        }
        return new DeterministicKey(this.getPath(), this.getChainCode(), this.pub, null, this.parent);
    }

    public DeterministicKey dropParent() {
        DeterministicKey key = new DeterministicKey(this.getPath(), this.getChainCode(), this.pub, this.priv, null);
        key.parentFingerprint = this.parentFingerprint;
        return key;
    }

    static byte[] addChecksum(byte[] input) {
        int inputLength = input.length;
        byte[] checksummed = new byte[inputLength + 4];
        System.arraycopy(input, 0, checksummed, 0, inputLength);
        byte[] checksum = Sha256Hash.hashTwice(input);
        System.arraycopy(checksum, 0, checksummed, inputLength, 4);
        return checksummed;
    }

    @Override
    public DeterministicKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        throw new UnsupportedOperationException("Must supply a new parent for encryption");
    }

    public DeterministicKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey, @Nullable DeterministicKey newParent) throws KeyCrypterException {
        byte[] privKeyBytes;
        Preconditions.checkNotNull((Object)keyCrypter);
        if (newParent != null) {
            Preconditions.checkArgument((boolean)newParent.isEncrypted());
        }
        Preconditions.checkState(((privKeyBytes = this.getPrivKeyBytes()) != null ? 1 : 0) != 0, (Object)"Private key is not available");
        EncryptedData encryptedPrivateKey = keyCrypter.encrypt(privKeyBytes, aesKey);
        DeterministicKey key = new DeterministicKey(this.childNumberPath, this.chainCode, keyCrypter, this.pub, encryptedPrivateKey, newParent);
        if (newParent == null) {
            key.setCreationTimeSeconds(this.getCreationTimeSeconds());
        }
        return key;
    }

    @Override
    public boolean isPubKeyOnly() {
        return super.isPubKeyOnly() && (this.parent == null || this.parent.isPubKeyOnly());
    }

    @Override
    public boolean hasPrivKey() {
        return this.findParentWithPrivKey() != null;
    }

    @Override
    @Nullable
    public byte[] getSecretBytes() {
        return this.priv != null ? this.getPrivKeyBytes() : null;
    }

    @Override
    public boolean isEncrypted() {
        return this.priv == null && (super.isEncrypted() || this.parent != null && this.parent.isEncrypted());
    }

    @Override
    @Nullable
    public KeyCrypter getKeyCrypter() {
        if (this.keyCrypter != null) {
            return this.keyCrypter;
        }
        if (this.parent != null) {
            return this.parent.getKeyCrypter();
        }
        return null;
    }

    @Override
    public ECKey.ECDSASignature sign(Sha256Hash input, @Nullable KeyParameter aesKey) throws KeyCrypterException {
        if (this.isEncrypted()) {
            return super.sign(input, aesKey);
        }
        BigInteger privateKey = this.findOrDerivePrivateKey();
        if (privateKey == null) {
            throw new ECKey.MissingPrivateKeyException();
        }
        return super.doSign(input, privateKey);
    }

    @Override
    public DeterministicKey decrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        BigInteger privKey = this.findOrDeriveEncryptedPrivateKey(keyCrypter, aesKey);
        DeterministicKey key = new DeterministicKey(this.childNumberPath, this.chainCode, privKey, this.parent);
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException.PublicPrivateMismatch("Provided AES key is wrong");
        }
        if (this.parent == null) {
            key.setCreationTimeSeconds(this.getCreationTimeSeconds());
        }
        return key;
    }

    @Override
    public DeterministicKey decrypt(KeyParameter aesKey) throws KeyCrypterException {
        return (DeterministicKey)super.decrypt(aesKey);
    }

    private BigInteger findOrDeriveEncryptedPrivateKey(KeyCrypter keyCrypter, KeyParameter aesKey) {
        if (this.encryptedPrivateKey != null) {
            byte[] decryptedKey = keyCrypter.decrypt(this.encryptedPrivateKey, aesKey);
            if (decryptedKey.length != 32) {
                throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + decryptedKey.length);
            }
            return new BigInteger(1, decryptedKey);
        }
        DeterministicKey cursor = this.parent;
        while (cursor != null && cursor.encryptedPrivateKey == null) {
            cursor = cursor.parent;
        }
        if (cursor == null) {
            throw new KeyCrypterException("Neither this key nor its parents have an encrypted private key");
        }
        byte[] parentalPrivateKeyBytes = keyCrypter.decrypt(cursor.encryptedPrivateKey, aesKey);
        if (parentalPrivateKeyBytes.length != 32) {
            throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + parentalPrivateKeyBytes.length);
        }
        return this.derivePrivateKeyDownwards(cursor, parentalPrivateKeyBytes);
    }

    private DeterministicKey findParentWithPrivKey() {
        DeterministicKey cursor = this;
        while (cursor != null && cursor.priv == null) {
            cursor = cursor.parent;
        }
        return cursor;
    }

    @Nullable
    private BigInteger findOrDerivePrivateKey() {
        DeterministicKey cursor = this.findParentWithPrivKey();
        if (cursor == null) {
            return null;
        }
        return this.derivePrivateKeyDownwards(cursor, cursor.priv.toByteArray());
    }

    private BigInteger derivePrivateKeyDownwards(DeterministicKey cursor, byte[] parentalPrivateKeyBytes) {
        DeterministicKey downCursor = new DeterministicKey(cursor.childNumberPath, cursor.chainCode, cursor.pub, new BigInteger(1, parentalPrivateKeyBytes), cursor.parent);
        ImmutableList path = this.childNumberPath.subList(cursor.getPath().size(), this.childNumberPath.size());
        for (ChildNumber num : path) {
            downCursor = HDKeyDerivation.deriveChildKey(downCursor, num);
        }
        if (!downCursor.pub.equals(this.pub)) {
            throw new KeyCrypterException.PublicPrivateMismatch("Could not decrypt bytes");
        }
        return (BigInteger)Preconditions.checkNotNull((Object)downCursor.priv);
    }

    public DeterministicKey derive(int child) {
        return HDKeyDerivation.deriveChildKey(this, new ChildNumber(child, true));
    }

    @Override
    public BigInteger getPrivKey() {
        BigInteger key = this.findOrDerivePrivateKey();
        Preconditions.checkState((key != null ? 1 : 0) != 0, (Object)"Private key bytes not available");
        return key;
    }

    @Deprecated
    public byte[] serializePublic(NetworkParameters params) {
        return this.serialize(params, true, Script.ScriptType.P2PKH);
    }

    @Deprecated
    public byte[] serializePrivate(NetworkParameters params) {
        return this.serialize(params, false, Script.ScriptType.P2PKH);
    }

    private byte[] serialize(NetworkParameters params, boolean pub, Script.ScriptType outputScriptType) {
        ByteBuffer ser = ByteBuffer.allocate(78);
        if (outputScriptType == Script.ScriptType.P2PKH) {
            ser.putInt(pub ? params.getBip32HeaderP2PKHpub() : params.getBip32HeaderP2PKHpriv());
        } else if (outputScriptType == Script.ScriptType.P2WPKH) {
            ser.putInt(pub ? params.getBip32HeaderP2WPKHpub() : params.getBip32HeaderP2WPKHpriv());
        } else {
            throw new IllegalStateException(outputScriptType.toString());
        }
        ser.put((byte)this.getDepth());
        ser.putInt(this.getParentFingerprint());
        ser.putInt(this.getChildNumber().i());
        ser.put(this.getChainCode());
        ser.put(pub ? this.getPubKey() : this.getPrivKeyBytes33());
        Preconditions.checkState((ser.position() == 78 ? 1 : 0) != 0);
        return ser.array();
    }

    public String serializePubB58(NetworkParameters params, Script.ScriptType outputScriptType) {
        return DeterministicKey.toBase58(this.serialize(params, true, outputScriptType));
    }

    public String serializePrivB58(NetworkParameters params, Script.ScriptType outputScriptType) {
        return DeterministicKey.toBase58(this.serialize(params, false, outputScriptType));
    }

    public String serializePubB58(NetworkParameters params) {
        return this.serializePubB58(params, Script.ScriptType.P2PKH);
    }

    public String serializePrivB58(NetworkParameters params) {
        return this.serializePrivB58(params, Script.ScriptType.P2PKH);
    }

    static String toBase58(byte[] ser) {
        return Base58.encode(DeterministicKey.addChecksum(ser));
    }

    public static DeterministicKey deserializeB58(String base58, NetworkParameters params) {
        return DeterministicKey.deserializeB58(null, base58, params);
    }

    public static DeterministicKey deserializeB58(@Nullable DeterministicKey parent, String base58, NetworkParameters params) {
        return DeterministicKey.deserialize(params, Base58.decodeChecked(base58), parent);
    }

    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey) {
        return DeterministicKey.deserialize(params, serializedKey, null);
    }

    public static DeterministicKey deserialize(NetworkParameters params, byte[] serializedKey, @Nullable DeterministicKey parent) {
        Object path;
        boolean priv;
        ByteBuffer buffer = ByteBuffer.wrap(serializedKey);
        int header = buffer.getInt();
        boolean pub = header == params.getBip32HeaderP2PKHpub() || header == params.getBip32HeaderP2WPKHpub();
        boolean bl = priv = header == params.getBip32HeaderP2PKHpriv() || header == params.getBip32HeaderP2WPKHpriv();
        if (!pub && !priv) {
            throw new IllegalArgumentException("Unknown header bytes: " + DeterministicKey.toBase58(serializedKey).substring(0, 4));
        }
        int depth = buffer.get() & 0xFF;
        int parentFingerprint = buffer.getInt();
        int i = buffer.getInt();
        ChildNumber childNumber = new ChildNumber(i);
        if (parent != null) {
            if (parentFingerprint == 0) {
                throw new IllegalArgumentException("Parent was provided but this key doesn't have one");
            }
            if (parent.getFingerprint() != parentFingerprint) {
                throw new IllegalArgumentException("Parent fingerprints don't match");
            }
            path = HDUtils.append(parent.getPath(), childNumber);
            if (path.size() != depth) {
                throw new IllegalArgumentException("Depth does not match");
            }
        } else {
            path = depth >= 1 ? ImmutableList.of((Object)childNumber) : ImmutableList.of();
        }
        byte[] chainCode = new byte[32];
        buffer.get(chainCode);
        byte[] data = new byte[33];
        buffer.get(data);
        Preconditions.checkArgument((!buffer.hasRemaining() ? 1 : 0) != 0, (Object)"Found unexpected data in key");
        if (pub) {
            return new DeterministicKey((ImmutableList<ChildNumber>)path, chainCode, new LazyECPoint(ECKey.CURVE.getCurve(), data), parent, depth, parentFingerprint);
        }
        return new DeterministicKey((ImmutableList<ChildNumber>)path, chainCode, new BigInteger(1, data), parent, depth, parentFingerprint);
    }

    @Override
    public long getCreationTimeSeconds() {
        if (this.parent != null) {
            return this.parent.getCreationTimeSeconds();
        }
        return super.getCreationTimeSeconds();
    }

    @Override
    public void setCreationTimeSeconds(long newCreationTimeSeconds) {
        if (this.parent != null) {
            throw new IllegalStateException("Creation time can only be set on root keys.");
        }
        super.setCreationTimeSeconds(newCreationTimeSeconds);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DeterministicKey other = (DeterministicKey)o;
        return super.equals(other) && Arrays.equals(this.chainCode, other.chainCode) && Objects.equal(this.childNumberPath, other.childNumberPath);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{super.hashCode(), Arrays.hashCode(this.chainCode), this.childNumberPath});
    }

    @Override
    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.add("pub", (Object)Utils.HEX.encode(this.pub.getEncoded()));
        helper.add("chainCode", (Object)Utils.HEX.encode(this.chainCode));
        helper.add("path", (Object)this.getPathAsString());
        if (this.creationTimeSeconds > 0L) {
            helper.add("creationTimeSeconds", this.creationTimeSeconds);
        }
        helper.add("isEncrypted", this.isEncrypted());
        helper.add("isPubKeyOnly", this.isPubKeyOnly());
        return helper.toString();
    }

    @Override
    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable KeyParameter aesKey, StringBuilder builder, NetworkParameters params, Script.ScriptType outputScriptType, @Nullable String comment) {
        builder.append("  addr:").append(Address.fromKey(params, this, outputScriptType).toString());
        builder.append("  hash160:").append(Utils.HEX.encode(this.getPubKeyHash()));
        builder.append("  (").append(this.getPathAsString());
        if (comment != null) {
            builder.append(", ").append(comment);
        }
        builder.append(")\n");
        if (includePrivateKeys) {
            builder.append("  ").append(this.toStringWithPrivate(aesKey, params)).append("\n");
        }
    }
}

